/************************************************************************************************************\

Module Name:    RtspClientModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "RtspClientModule.h"
#include "CommonUtils.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------
typedef struct tagStrmThreadData
{
    RtspClientModule*   poRtspClnMod;
    MBOOL               bIsMainThread;
    MBOOL               bIsVideoThread;
} StrmThreadData;

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "RtspCln";
static          MUINT32     g_uiRtspClnModCount     = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       RtspClnMod_EventCallback

Description:    .

\************************************************************************************************************/
static LStatus RtspClnMod_EventCallback(
            void*                   pvEventContext,
            LNetStreamer_Handle     hNetStreamer,
            LNetStreamer_EventType* peEventType)
{
    LStatus eStatus;

    (void)(pvEventContext);
    (void)(hNetStreamer);

    if (LNetStreamer_EventType_FRAME_SKIP == *peEventType)
    {
        MsgLog(0, "**** DETECTED FRAME SKIP!  *****");
    }

    eStatus = LStatus_OK;

    return eStatus;
}

/************************************************************************************************************\

Function:       RtspClnMod_CreateSession

Description:    .

\************************************************************************************************************/
LStatus RtspClnMod_CreateSession(RtspClientModule* poRtspClnMod, AudioSpecificCfg* poAudioCfg)
{
    if (poRtspClnMod == MNULL)
    {
        return LStatus_INVALID_PARAM;
    }

    LNetStreamer_Config* poSessCfg = &(poRtspClnMod->oConfig);

    LStatus eStatus = LNetStreamer_Create(&poSessCfg->eConfigType, &(poRtspClnMod->hSession));

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        MUINT32 uiConfigLen;

        eStatus = LNetStreamer_GetConfigLength(
                        poRtspClnMod->hSession,
                        LNetStreamer_ConfigType_STANDARD,
                        &uiConfigLen);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            LNetStreamer_Config* poConfig = malloc(uiConfigLen);

            if (poConfig != MNULL)
            {
                memset(poConfig, 0, uiConfigLen);

                poConfig->eConfigType = LNetStreamer_ConfigType_STANDARD;

                eStatus = LNetStreamer_GetConfig(poRtspClnMod->hSession, &poConfig->eConfigType, uiConfigLen);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    // TODO: Extract stream infos from SPS/PPS when a SPS/PPS will be available and
                    //       LH264D services will be ready.
                    MsgLog(2, "SPS= %p, len= %u, PPS= %p, len= %u",
                              poConfig->paucVideoHeaderSps,
                              poConfig->uiVideoHeaderSpsLength,
                              poConfig->paucVideoHeaderPps,
                              poConfig->uiVideoHeaderPpsLength);

                    if (poSessCfg->bEnableAudio)
                    {
                        if(poConfig->uiAudioHeaderLength == 2)
                        {
                            poAudioCfg->bValid  = MTRUE;
                            poAudioCfg->uiCfg   = (poConfig->paucAudioHeader[1] << 8)
                                                    + (poConfig->paucAudioHeader[0]);

                            MsgLog(2, "AudioSpecificConfig: %02x%02x",
                                    poConfig->paucAudioHeader[0],
                                    poConfig->paucAudioHeader[1]);
                        }
                        else if (poConfig->eProtocol == LNetStreamer_Protocol_RTSP)
                        {
                            MsgLog(0, "WARNING: no AudioSpecificConfig! (AudioHeader=%p, Length=%u)\n",
                                   poConfig->paucAudioHeader, poConfig->uiAudioHeaderLength);
                        }
                    }
                }

                free(poConfig);
            }
            else
            {
                eStatus = LStatus_OUT_OF_MEMORY;
            }
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       RtspClnMod_DestroySession

Description:    .

\************************************************************************************************************/
void RtspClnMod_DestroySession(RtspClientModule* poRtspClnMod)
{
    if ((poRtspClnMod != MNULL)
         && (poRtspClnMod->hSession != MNULL))
    {
        MUINT i;

        for (i = 0; i < poRtspClnMod->oVideoOutLink.uiBufferCount; i++)
        {
            BufferInfo* poBufferInfo = &(poRtspClnMod->oVideoOutLink.aoBufferInfo[i]);

            if ((poBufferInfo->hBuffer != MNULL)
                && (poBufferInfo->pvPrivateData != MNULL))
            {
                LNetStreamer_ReleaseMedia(
                            poRtspClnMod->hSession,
                            (LNetStreamer_MediaType*)(poBufferInfo->pvPrivateData));

                poBufferInfo->hBuffer = MNULL;
            }
        }

        for (i = 0; i < poRtspClnMod->oAudioOutLink.uiBufferCount; i++)
        {
            BufferInfo* poBufferInfo = &(poRtspClnMod->oAudioOutLink.aoBufferInfo[i]);

            if ((poBufferInfo->hBuffer != MNULL)
                && (poBufferInfo->pvPrivateData != MNULL))
            {
                LNetStreamer_ReleaseMedia(
                            poRtspClnMod->hSession,
                            (LNetStreamer_MediaType*)(poBufferInfo->pvPrivateData));

                poBufferInfo->hBuffer = MNULL;
            }
        }

        LNetStreamer_Destroy(poRtspClnMod->hSession);
        poRtspClnMod->hSession = MNULL;
    }
}

/************************************************************************************************************\

Function:       RtspClnMod_Init

Description:    .

\************************************************************************************************************/
LStatus RtspClnMod_Init(
            RtspClientModule*       poRtspClnMod,
            LDevice_Handle          hDevice,
            MBOOL                   bEnableVideo,
            MBOOL                   bEnableAudio,
            MUINT                   uiVideoBufferCount,
            MUINT                   uiVideoBufferSizeBytes,
            MUINT                   uiAudioBufferCount,
            MUINT                   uiAudioBufferSizeBytes,
            LNetStreamer_Protocol   eProtocol,
            const char*             szLocation,
            const char*             szAddress,
            MUINT                   uiVideoPort,
            MUINT                   uiAudioPort,
            MBOOL                   bSkipToNextIdr,
            MUINT                   uiAudioSampleRate,
            MUINT                   uiMtu,
            MBOOL                   bEnableSrt,
            const char*             szNetInterface,
            MBOOL                   bEnableIpv6,
            LNetStreamer_SrtMode    eSrtMode)
{
    MsgLog(2, "{...");

    RtspClnMod_Cleanup(poRtspClnMod);

    LStatus eStatus = ((poRtspClnMod == MNULL)
                       || (hDevice == MNULL)
                       || !(bEnableVideo || bEnableAudio)
                       || (bEnableVideo
                            && (uiVideoBufferSizeBytes == 0))
                       || (bEnableAudio
                            && (uiAudioBufferSizeBytes == 0))
                       || (bEnableAudio
                            && (eProtocol == LNetStreamer_Protocol_RTP)
                            && (uiAudioSampleRate == 0)))
                      ? LStatus_INVALID_PARAM : LStatus_OK;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        snprintf(
                    poRtspClnMod->szModuleName,
                    sizeof(poRtspClnMod->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    g_uiRtspClnModCount);
    }

    if (bEnableVideo && LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = ModLnk_Init(
                    &(poRtspClnMod->oVideoOutLink),
                    hDevice,
                    uiVideoBufferCount,
                    MNULL,
                    MTRUE,
                    sizeof(LNetStreamer_Media),
                    poRtspClnMod->szModuleName);
    }

    if (bEnableAudio && LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = ModLnk_Init(
                    &(poRtspClnMod->oAudioOutLink),
                    hDevice,
                    uiAudioBufferCount,
                    MNULL,
                    MTRUE,
                    sizeof(LNetStreamer_Media),
                    poRtspClnMod->szModuleName);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LNetStreamer_Config* poSessCfg = &(poRtspClnMod->oConfig);

        strncpy_wz(poRtspClnMod->szLocation, szLocation, sizeof(poRtspClnMod->szLocation));
        memset(poSessCfg, 0, sizeof(*poSessCfg));

        if (eProtocol == LNetStreamer_Protocol_TS)
        {
            // Only video port is taken into account with TS.
            if (!bEnableVideo)
            {
                uiVideoPort = uiAudioPort;
            }

            uiAudioPort = 0;
        }

        poSessCfg->eConfigType         = LNetStreamer_ConfigType_STANDARD;
        poSessCfg->eService            = LNetStreamer_Service_CLIENT;
        poSessCfg->eProtocol           = eProtocol;
        poSessCfg->hDevice             = hDevice;
        poSessCfg->bEnableVideo        = bEnableVideo;
        poSessCfg->uiVideoBufferCount  = bEnableVideo ? uiVideoBufferCount : 0;
        poSessCfg->uiVideoBufferLength = bEnableVideo ? uiVideoBufferSizeBytes : 0;
        poSessCfg->bEnableAudio        = bEnableAudio;
        poSessCfg->uiAudioBufferCount  = bEnableAudio ? uiAudioBufferCount : 0;
        poSessCfg->uiAudioBufferLength = bEnableAudio ? uiAudioBufferSizeBytes : 0;

        if ((LNetStreamer_Protocol_RTSP == eProtocol) ||
            (LNetStreamer_Protocol_RTMP == eProtocol))
        {
            poSessCfg->pacLocation         = poRtspClnMod->szLocation;
            poSessCfg->uiLocationLength    = strlen(poRtspClnMod->szLocation);
            poSessCfg->uiConfigFlags       |= LNetStreamer_ConfigFlags_LOCATION;
        }

        if (1 == uiVideoPort)
        {
            poSessCfg->uiConfigFlags   |= LNetStreamer_ConfigFlags_EXTRA_FLAGS;
            poSessCfg->uiConfigFlags2  |= LNetStreamer_ConfigFlags2_TCP_INTERLEAVED;
        }
        else if (uiVideoPort)
        {
            poSessCfg->uiConfigFlags   |= LNetStreamer_ConfigFlags_VIDEO_PORT;
            poSessCfg->uiVideoPort     = uiVideoPort;
        }

        if (uiAudioPort && bEnableAudio)
        {
            poSessCfg->uiConfigFlags   |= LNetStreamer_ConfigFlags_AUDIO_PORT;
            poSessCfg->uiAudioPort     = uiAudioPort;
        }

        if (ENABLE_AUTO_SYNC)
        {
            poSessCfg->uiConfigFlags |= LNetStreamer_ConfigFlags_LBUFFER_AUTO_SYNC;
        }

        if (szAddress)
        {
            strncpy_wz(poRtspClnMod->szMcastAddr, szAddress, sizeof(poRtspClnMod->szMcastAddr));

            poSessCfg->uiConfigFlags          |= LNetStreamer_ConfigFlags_DESTINATION;
            poSessCfg->pacDestination         = poRtspClnMod->szMcastAddr;
            poSessCfg->uiDestinationLength    = strlen(poRtspClnMod->szMcastAddr);
        }

        if (1)
        {
            poSessCfg->uiConfigFlags     |= LNetStreamer_ConfigFlags_RTSP_SERVER
                                              | LNetStreamer_ConfigFlags_RTMP_SERVER
                                              | LNetStreamer_ConfigFlags_RTSP_PORT;
            poSessCfg->bEnableRtspServer  = MTRUE;
            poSessCfg->bEnableRtmpServer  = MTRUE;
            poSessCfg->uiRtspPort         = 3049;
        }

        if (!bSkipToNextIdr)
        {
            poSessCfg->uiConfigFlags |= LNetStreamer_ConfigFlags_NO_SKIP;
        }

        if (eProtocol == LNetStreamer_Protocol_RTP)
        {
            const char* szVideoSdpFmt
                            =  "m=video %u RTP/AVP 96\r\n"
                               "a=rtpmap:96 H264/90000\r\n";
            const char* szAudioSdpFmt
                            =  "m=audio %u RTP/AVP 97\r\n"
                               "a=rtpmap:97 mpeg4-generic/%u\r\n";
            const char* szSdpFmt
                            =  "v=0\r\n"
                               "%s"     // Video part
                               "%s";    // Audio part

            char szVideoSdp[128] = {0};
            char szAudioSdp[128] = {0};

            if (bEnableVideo)
            {
                snprintf(szVideoSdp, sizeof(szVideoSdp), szVideoSdpFmt, uiVideoPort);
            }

            if (bEnableAudio)
            {
                snprintf(szAudioSdp, sizeof(szAudioSdp), szAudioSdpFmt, uiAudioPort, uiAudioSampleRate);
            }

            snprintf(poRtspClnMod->szSdp, sizeof(poRtspClnMod->szSdp), szSdpFmt, szVideoSdp, szAudioSdp);

            poSessCfg->uiConfigFlags |= LNetStreamer_ConfigFlags_SDP;
            poSessCfg->pacSdp         = poRtspClnMod->szSdp;
            poSessCfg->uiSdpLength    = strlen(poRtspClnMod->szSdp);
        }

        poSessCfg->uiConfigFlags |= LNetStreamer_ConfigFlags_EVENT;
        poSessCfg->pfnEventCallback = &RtspClnMod_EventCallback;
        poSessCfg->pvEventContext = poRtspClnMod;

        poSessCfg->uiConfigFlags |= LNetStreamer_ConfigFlags_RTCP_FREQUENCY;
        poSessCfg->uiRtcpFrequency = 1000;

        if (uiMtu)
        {
            poSessCfg->uiConfigFlags |= LNetStreamer_ConfigFlags_MTU;
            poSessCfg->uiMtu = uiMtu;
        }

        if (bEnableSrt)
        {
            poSessCfg->uiConfigFlags |= LNetStreamer_ConfigFlags_SRT;

            if (eSrtMode)
            {
                poSessCfg->uiConfigFlags |= LNetStreamer_ConfigFlags_EXTRA_FLAGS;
                poSessCfg->uiConfigFlags2 |= LNetStreamer_ConfigFlags2_SRT_MODE;
                poSessCfg->eSrtMode = eSrtMode;
            }
        }

        if (szNetInterface)
        {
            strncpy_wz(poRtspClnMod->szNetInterface, szNetInterface, sizeof(poRtspClnMod->szNetInterface));

            poSessCfg->uiConfigFlags |= LNetStreamer_ConfigFlags_NET_INTERFACE;
            poSessCfg->pacNetInterface = poRtspClnMod->szNetInterface;
            poSessCfg->uiNetInterfaceLength = (MUINT32)(strlen(poRtspClnMod->szNetInterface));
        }

        if (bEnableIpv6)
        {
            poSessCfg->uiConfigFlags |= LNetStreamer_ConfigFlags_EXTRA_FLAGS;
            poSessCfg->uiConfigFlags2 |= LNetStreamer_ConfigFlags2_IPV6;
        }

        eStatus = RtspClnMod_CreateSession(poRtspClnMod, &(poRtspClnMod->oAudioCfg));

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            RtspClnMod_DestroySession(poRtspClnMod);
        }
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        poRtspClnMod->bEnableVideo = bEnableVideo;
        poRtspClnMod->bEnableAudio = bEnableAudio;
    }
    else
    {
        RtspClnMod_Cleanup(poRtspClnMod);
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       RtspClnMod_Cleanup

Description:    .

\************************************************************************************************************/
void RtspClnMod_Cleanup(RtspClientModule* poRtspClnMod)
{
    MsgLog(2, "{...");

    if (poRtspClnMod != MNULL)
    {
        RtspClnMod_DestroySession(poRtspClnMod);

        ModLnk_Cleanup(&(poRtspClnMod->oVideoOutLink));
        ModLnk_Cleanup(&(poRtspClnMod->oAudioOutLink));
        memset(&(poRtspClnMod->oAudioCfg), 0, sizeof(poRtspClnMod->oAudioCfg));
        memset(poRtspClnMod->szSdp, 0, sizeof(poRtspClnMod->szSdp));
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       RtspClnMod_GetNetBuffer

Description:    .

\************************************************************************************************************/
LStatus RtspClnMod_GetNetBuffer(
    RtspClientModule*       poRtspClnMod,
    BufferInfo*             poDstBuffer,
    LNetStreamer_MediaType  eMediaType,
    MUINT64*                puiLastBufferTime)
{
    LStatus eStatus = ((poDstBuffer->pvPrivateData != MNULL) && poDstBuffer->bInternal)
                        ? LStatus_OK : LStatus_FAIL;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LNetStreamer_Media* poNetBufInfo = (LNetStreamer_Media*)(poDstBuffer->pvPrivateData);

        if (poDstBuffer->hBuffer != MNULL)
        {
            MsgLog(4, "LNetStreamer_ReleaseBufferInfo: Buffer[%u].", poDstBuffer->uiId);

            eStatus = LNetStreamer_ReleaseMedia(
                            poRtspClnMod->hSession,
                            &poNetBufInfo->eMediaType);

            if (LSTATUS_IS_FAIL(eStatus))
            {
                MsgLogErr(
                    "ERROR! LNetStreamer_ReleaseBufferInfo returned status= %d (%s). Continue...",
                    eStatus,
                    GetStatusStr(eStatus));
            }

            poDstBuffer->hBuffer = MNULL;
        }

        poNetBufInfo->eMediaType = eMediaType;

        MsgLog(4, "LNetStreamer_GetMedia(Buffer[%u])...", poDstBuffer->uiId);

        eStatus = LNetStreamer_GetMedia(
                        poRtspClnMod->hSession,
                        &poNetBufInfo->eMediaType);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            MsgLog(4, "LNetStreamer_GetMedia(Buffer[%u]) done. (offset= %u, size= %u)",
                      poDstBuffer->uiId, poNetBufInfo->uiDataOffset, poNetBufInfo->uiDataLength);

            *puiLastBufferTime              = GetMonoTimeUsec();
            poDstBuffer->hBuffer            = poNetBufInfo->hBuffer;
            poDstBuffer->uiStartOffset      = poNetBufInfo->uiDataOffset;
            poDstBuffer->uiSizeBytes        = poNetBufInfo->uiDataLength;
            poDstBuffer->uiSyncPtsUsec      = (MUINT64)poNetBufInfo->uiPTS * 1000;
            poDstBuffer->uiTimestampUsec    = poDstBuffer->uiSyncPtsUsec;
        }
        else if (LStatus_TIMEOUT == eStatus)
        {
            MsgLog(4, "LNetStreamer_GetMedia(Buffer[%u]) skip.",
                      poDstBuffer->uiId);
        }
        else
        {
            MsgLog(4, "LNetStreamer_GetMedia(Buffer[%u]) ERROR! (status = %d - %s)",
                      poDstBuffer->uiId, eStatus, GetStatusStr(eStatus));

            // Interpret any errors as a timeout

            if ((GetMonoTimeUsec() - *puiLastBufferTime) > (10 * 1000 * 1000))
            {
                poDstBuffer->bEndOfStream = MTRUE;
                eStatus = LStatus_OK;
            }
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       RtspClnMod_StreamingThread

Description:    .

\************************************************************************************************************/
LStatus RtspClnMod_StreamingThread(void* pvData)
{
    StrmThreadData*     poThreadData    = (StrmThreadData*) pvData;
    RtspClientModule*   poRtspClnMod    = poThreadData->poRtspClnMod;
    ModuleThread*       poThread        = poThreadData->bIsMainThread
                                            ? &(poRtspClnMod->oMainCpuThread)
                                            : &(poRtspClnMod->oChildCpuThread);
    ModuleLink*         poOutLink       = (poThreadData->bIsVideoThread)
                                            ? &(poRtspClnMod->oVideoOutLink)
                                            : &(poRtspClnMod->oAudioOutLink);
    LNetStreamer_MediaType eMediaType   = (poThreadData->bIsVideoThread)
                                            ? (LNetStreamer_MediaType_VIDEO)
                                            : (LNetStreamer_MediaType_AUDIO);

    if(poThreadData->bIsVideoThread)
    {
        static MUINT32  uiRtspClnIndex = 0;

        MCHAR8  szThreadName[16] = "";

        snprintf(szThreadName, sizeof(szThreadName), "RtspClnVideo%d", uiRtspClnIndex++);

        ModThread_SetName(szThreadName);
    }
    else
    {
        static MUINT32  uiRtspClnAIndex = 0;

        MCHAR8  szThreadName[16] = "";

        snprintf(szThreadName, sizeof(szThreadName), "RtspClnAudio%d", uiRtspClnAIndex++);

        ModThread_SetName(szThreadName);
    }

    if(!poThreadData->bIsMainThread)
    {
        MsgLog(2, "Start thread %p\n", pthread_self());
    }

    MUINT64 uiLastBufferTime = GetMonoTimeUsec();

    while (!poThread->bKillThread)
    {
        BufferInfo* poDstBuffer = MNULL;

        // Get buffer to fill with media.
        LStatus eStatus = ModLnk_GetReturnedBuffer(poOutLink, 100, MNULL, &poDstBuffer);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            // Get media from network.
            eStatus = RtspClnMod_GetNetBuffer(
                            poRtspClnMod,
                            poDstBuffer,
                            eMediaType,
                            &uiLastBufferTime);

            if(poDstBuffer->bEndOfStream)
            {
                poThread->bKillThread           = MTRUE;
                poRtspClnMod->bLostConnection   = MTRUE;
                eStatus = LStatus_OK;
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                ModLnk_SubmitBuffer(poOutLink, poDstBuffer, MNULL, NO_TAG);
            }
            else
            {
                ModLnk_ReleaseBuffer(poOutLink, poDstBuffer);
            }
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       RtspClnMod_MainCpuThread

Description:    .

\************************************************************************************************************/
LStatus RtspClnMod_MainCpuThread(void* pvData)
{
    ModThread_SetName("RtspCln");

    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    MsgLog(2, "Start thread %p.", pthread_self());

    RtspClientModule* poRtspClnMod  = (RtspClientModule*)pvData;
    LStatus           eStatus       = LStatus_OK;

    StrmThreadData oAudioData;
    oAudioData.bIsMainThread = MTRUE;
    oAudioData.bIsVideoThread = MFALSE;
    oAudioData.poRtspClnMod = poRtspClnMod;

    if(poRtspClnMod->bEnableVideo)
    {
        if(poRtspClnMod->bEnableAudio)
        {
            oAudioData.bIsMainThread = MFALSE;
            eStatus = ModThread_Start(&(poRtspClnMod->oChildCpuThread), &oAudioData, RtspClnMod_StreamingThread);
        }

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            StrmThreadData oVideoData;
            oVideoData.bIsMainThread = MTRUE;
            oVideoData.bIsVideoThread = MTRUE;
            oVideoData.poRtspClnMod = poRtspClnMod;

            eStatus = RtspClnMod_StreamingThread(&oVideoData);
        }
    }
    else if(poRtspClnMod->bEnableAudio)
    {
        eStatus = RtspClnMod_StreamingThread(&oAudioData);
    }
    else
    {
        eStatus = LStatus_FAIL;
    }

    if(!oAudioData.bIsMainThread)
    {
        ModThread_Stop(&(poRtspClnMod->oChildCpuThread));
    }

    MsgLog(2, "Kill thread.");

    return eStatus;
}

/************************************************************************************************************\

Function:       RtspClnMod_Start

Description:    .

\************************************************************************************************************/
LStatus RtspClnMod_Start(RtspClientModule* poRtspClnMod)
{
    MsgLog(2, "{...");

    LStatus eStatus = LStatus_INVALID_PARAM;

    if (poRtspClnMod != MNULL)
    {
        MBOOL32 bGoodConnection = MTRUE;

        if((poRtspClnMod->bEnableVideo && (poRtspClnMod->oVideoOutLink.uiSubmitCount == 0))
            || (poRtspClnMod->bEnableAudio && (poRtspClnMod->oAudioOutLink.uiSubmitCount == 0)))
        {
            bGoodConnection = MFALSE;
        }

        if (bGoodConnection)
        {
            if (poRtspClnMod->hSession == MNULL)
            {
                AudioSpecificCfg oAudioCfg = {0};

                eStatus = RtspClnMod_CreateSession(poRtspClnMod, &oAudioCfg);

                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    if ((oAudioCfg.bValid != poRtspClnMod->oAudioCfg.bValid)
                         || (oAudioCfg.uiCfg != poRtspClnMod->oAudioCfg.uiCfg))
                    {
                        MsgLogErr("ERROR! Cannot start '%s': Some audio stream parameters changed.",
                                  poRtspClnMod->oConfig.pacLocation);

                        RtspClnMod_DestroySession(poRtspClnMod);
                        eStatus = LStatus_FAIL;
                    }
                }
            }

            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                poRtspClnMod->bLostConnection = MFALSE;

                eStatus = ModThread_Start(&(poRtspClnMod->oMainCpuThread), poRtspClnMod, RtspClnMod_MainCpuThread);
            }
        }
        else
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       RtspClnMod_Stop

Description:    .

\************************************************************************************************************/
void RtspClnMod_Stop(RtspClientModule* poRtspClnMod)
{
    MsgLog(2, "{...");

    if (poRtspClnMod != MNULL)
    {
        ModThread_Stop(&(poRtspClnMod->oMainCpuThread));
    }

    MsgLog(2, "...}");
}

